<?php

namespace App\Http\Controllers;

use App\Models\EducationBackground;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\ValidationException;
use Exception;

class EducationBackgroundController extends Controller
{
    public function index(): JsonResponse
    {
        try {
            $education = EducationBackground::with('user')->get();
            return response()->json(['success' => true, 'data' => $education, 'message' => 'Education background retrieved successfully'], 200);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to retrieve education background: ' . $e->getMessage()], 500);
        }
    }

    public function store(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'User_id' => 'nullable|exists:users,User_id',
                'Institution' => 'required|string|max:255',
                'Degree' => 'required|string|max:255',
                'Field_of_study' => 'required|string|max:255',
                'Start_date' => 'required|date',
                'End_date' => 'nullable|date|after_or_equal:Start_date',
                'Currently_studying' => 'boolean',
                'Description' => 'nullable|string|max:65535',
            ]);

            $education = EducationBackground::create($validated);
            $education->load('user');
            return response()->json(['success' => true, 'data' => $education, 'message' => 'Education background created successfully'], 201);
        } catch (ValidationException $e) {
            return response()->json(['success' => false, 'message' => 'Validation failed', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to create education background: ' . $e->getMessage()], 500);
        }
    }

    public function show($id): JsonResponse
    {
        try {
            $education = EducationBackground::with('user')->findOrFail($id);
            return response()->json(['success' => true, 'data' => $education, 'message' => 'Education background retrieved successfully'], 200);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Education background not found: ' . $e->getMessage()], 404);
        }
    }

    public function update(Request $request, $id): JsonResponse
    {
        try {
            $education = EducationBackground::findOrFail($id);
            $validated = $request->validate([
                'User_id' => 'nullable|exists:users,User_id',
                'Institution' => 'required|string|max:255',
                'Degree' => 'required|string|max:255',
                'Field_of_study' => 'required|string|max:255',
                'Start_date' => 'required|date',
                'End_date' => 'nullable|date|after_or_equal:Start_date',
                'Currently_studying' => 'boolean',
                'Description' => 'nullable|string|max:65535',
            ]);

            $education->update($validated);
            $education->load('user');
            return response()->json(['success' => true, 'data' => $education, 'message' => 'Education background updated successfully'], 200);
        } catch (ValidationException $e) {
            return response()->json(['success' => false, 'message' => 'Validation failed', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to update education background: ' . $e->getMessage()], 500);
        }
    }

    public function destroy($id): JsonResponse
    {
        try {
            $education = EducationBackground::findOrFail($id);
            $education->delete();
            return response()->json(['success' => true, 'data' => null, 'message' => 'Education background deleted successfully'], 204);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to delete education background: ' . $e->getMessage()], 500);
        }
    }
}
